<?php

namespace Cloudflare;

class Turnstile {

	private $secret;

	public function __construct( $secret ) {
		$this->secret = $secret;
	}

	public function verify( $token ) {

		// Retrieve the Turnstile response token
		$ip = isset( $_SERVER['REMOTE_ADDR'] ) ? $_SERVER['REMOTE_ADDR'] : '';

		// Validate the token by calling the "/siteverify" API endpoint.
		$url = 'https://challenges.cloudflare.com/turnstile/v0/siteverify';

		$data = [
			'secret'   => $this->secret,
			'response' => $token,
			'remoteip' => $ip
		];

		$options = [
			'http' => [
				'header'  => "Content-Type: application/json\r\n",
				'method'  => 'POST',
				'content' => json_encode( $data )
			]
		];

		try {
			$context = stream_context_create( $options );
			$result  = file_get_contents( $url, false, $context );

			if ( $result !== false ) {
				$result_array = json_decode( $result, true );

				if ( $result_array['success'] === false && ! empty( $result_array['error-codes'] ) ) {
					// Verification failed
					foreach ( $result_array['error-codes'] as $code ) {
						\SwpmLog::log_simple_debug( "[Cloudflare Turnstile Addon] " . $this->error_msg( $code ), false );
					}
					return false;
				}

				// Verification successful
				\SwpmLog::log_simple_debug( "[Cloudflare Turnstile Addon] Turnstile captcha validated successfully!", true );

				return true;
			} else {
				\SwpmLog::log_simple_debug( "[Cloudflare Turnstile Addon] Error contacting verification server.", false );

				return false;
			}
		} catch ( \Exception $e){
			wp_die( $e->getMessage() );
		}

	}

	private function error_msg( $code ) {
		switch ( $code ) {
			case 'missing-input-secret':
				return __( 'The secret parameter was not passed.', 'simple-membership' );
			case 'invalid-input-secret':
				return __( 'The secret parameter was invalid, did not exist, or is a testing secret key with a non-testing response.', 'simple-membership' );
			case 'missing-input-response':
				return __( 'The response parameter (token) was not passed.', 'simple-membership' );
			case 'invalid-input-response':
				return __( 'The response parameter (token) is invalid or has expired.', 'simple-membership' );
			case 'bad-request':
				return __( 'The request was rejected because it was malformed.', 'simple-membership' );
			case 'timeout-or-duplicate':
				return __( 'The response parameter (token) has already been validated before.', 'simple-membership' );
			case 'internal-error':
				return __( 'An internal error happened while validating the response. The request can be retried.', 'simple-membership' );
			default:
				return '';
		}
	}
}